/*
 * HD Radio Controller
 * (c) Hal Vaughan 2008
 * hal@halblog.com
 * licensed under the Free Software Foundations General Public License 2.0
 *
 * This program makes it possible to control several different HD and satellite
 * radios from Linux.  It can be used for simple command line control as well as
 * control from a more sophisticated program using this interface as a library.
 *
 * Control protocols provided by Paul Cotter.
 */

#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <time.h>
#include "hdlinuxio.h"
#include "halcyon.h"
using namespace std;

//Leave these comments in place -- the original author uses them with a
//Perl script to generate headers

//LinuxPort::	bool isOpen, verbose;
//LinuxPort::	int portfd, lastread, cflag;
//LinuxPort::	float autodiscoverwait, defautodiscwait;
//LinuxPort::	unsigned int testparm[11];
//LinuxPort::	unsigned long naptime;
//LinuxPort::	long BAUD;
//LinuxPort::	long DATABITS;
//LinuxPort::	long STOPBITS;
//LinuxPort::	long PARITYON;
//LinuxPort::	long PARITY;
// Default value: use the RS232 port
//LinuxPort::	string serialdevice;
//LinuxPort::	string testdata;

//LinuxPort::	vector<string> devlist;
//LinuxPort::	vector<string> devtype;

//LinuxPort::	struct termios options, oldios;

//LinuxPort::	ConfigFile* mainConfig;

/**
 * Create the LinuxPort class.  Set all initial variables.
 */
LinuxPort::LinuxPort() {
	isOpen = false;
	portfd = 0; lastread = 0; cflag = 0xA4;
	testparm[0] = 0xA4; testparm[1] = 0x08;  testparm[2] = 0x01;  testparm[3] = 0x00;
	testparm[4] = 0x02; testparm[5] = 0x00;  testparm[6] = 0x01;  testparm[7] = 0x00;
	testparm[8] = 0x00;  testparm[9] = 0x00; testparm[10] = 0xB0;
//DEBUG: larger number makes debugging easier, smaller makes it respond faster
	naptime = 100;
	BAUD = B115200;
	DATABITS = CS8;
	STOPBITS = 0;
	PARITYON = 0;
	PARITY = 0;
	serialdevice = ""; testdata = "";
	verbose = false;
	defautodiscwait = 10;
//DEBUG:
// 	verbose = true;
}

/**
 * Add the config object.  Must be done!! Config is needed to supply some variables
 * and to make sure the serial port device is saved for use next time.
 * @param cf config object found in halcyon.cpp
 */
void LinuxPort::setconfig(ConfigFile* cf) {//public
	string sval;
	mainConfig = cf;
	sval = mainConfig->getval("autodiscoverwait");
	if (sval != "") {
		sscanf(sval.c_str(), "%f", &autodiscoverwait);
	} else {
		autodiscoverwait = defautodiscwait;
	}
	return;
}

/**
 * Set the verbosity.  Provides output to the console for debugging.
 * @param verbosity true for console output
 */
void LinuxPort::setverbose(bool verbosity) {//public
	verbose = verbosity;
	return;
}

/**
 * Test a port to see if a radio we can work with is connected.  We open it
 * with openport() and wait to see if that turned on a radio that
 * sends us a power reply.  Checking a port could disrupt communications with
 * any other device on that port, so this is not recommended for use each time
 * a program is run but only the first time, to find the radio.
 * @param serport the serial port to check
 */
bool LinuxPort::testport(string serport) { //public
	unsigned char cIn;
	char* buff = new char[5];
	int i = 0, rd;
	double itime;
	float adjwait;		//Wait up to this many seconds for signal
	bool result, doreply = false;
	time_t tstart, tnow;

	cout << "Testing port for HD Radio Control: " << serport << endl;
	if (serport == "") {
		return false;
	}
	adjwait = autodiscoverwait;
	if (verbose) cout << "Wait time set: " << adjwait;
	setserialport(serport);
	result = openport();
	if (!result) {
		cout << "Error opening port: " << serport << endl;
		cout << "Cannot verify HD Radio on port: " << serport << endl;
		closeport();
		return false;
	}
	fcntl(portfd, F_SETFL, FNDELAY);
	time(&tstart);
	while (true) {
		rd = read(portfd, buff, 1);
		time(&tnow);
		itime = difftime(tnow, tstart);
// 		if (verbose) cout << "\tWait time, in seconds: " << itime << endl;
		if (itime >= adjwait) {
			cout << "Response time too long.  Port failed: " << serport << endl;
			cout << "Cannot verify HD Radio on port: " << serport << endl;
			closeport();
			return false;
		}
		usleep(naptime);
		if (rd < 1) continue;
		cIn = buff[0];
		chout(cIn);
		adjwait += .5;
		if (verbose) cout << "Got bytes from port!, New Wait time: " << adjwait << endl;
		if (doreply) {
			if (verbose) cout << "\tComparing to parm #: " << i << endl;
			if (cIn != testparm[i]) {
// 				cout << "Bad byte in sequence, count: " << i << ", Byte: " << testparm[i] << endl;
				cout << "Cannot verify HD Radio on port: " << serport << endl;
				closeport();
				return false;
			}
			i++;
// 			adjwait += .5;
// 			cout << "\tCurrent count: " << i << ", Size: " << sizeof(testparm)/sizeof(i) << endl;
			if (i == sizeof(testparm)/sizeof(int)) {
				cout << "Port matched for HD Radio: " << serport << endl;
				fcntl(portfd, F_SETFL, 0);
				return true;
			}

		} else if (cIn == cflag) {
			if (verbose) cout << "We have the first command byte\n";
			doreply = true;
			i++;
		}
	}
	cout << "Cannot verify HD Radio on port: " << serport << endl;
	closeport();
	return false;
}

/**
 * Find the HD Radio hooked up to this system.  Also referred to as autodiscovery.  This should
 * not be done unless the radio is not found or can't be confirmed.  This uses the testport() function
 * which can disrupt communications with other devices on serial ports while we search for a compatible
 * radio.  For that reason, autodiscovery should only be used the first time a program is ever run to find
 * the radio or when it has been moved to a different port.  If the port is known, autodiscovery can be
 * avoided just by specifying what port to use and setting the firstrun variable in the config file to
 * false.
 *  @return true if a radio was found
 */
bool LinuxPort::findhdradio() { //public
	bool result = false, nonexist = true, opt = false;
	unsigned int ispec = 0, iopt = 0;
	int ocount = 0, fstat, ival;
	string serdev = serialdevice, testdev, sval;
	struct stat finfo;

	sval = mainConfig->getval("autodiscoverwait");
	sscanf(sval.c_str(), "%d", &ival);
	autodiscoverwait = ival;
	cout << "Auto discover wait: " << autodiscoverwait << endl;

// 	result = testport(serdev);
// 	if (result) return true;
	if (devlist.empty())
		devlist.push_back("");
	if (devtype.empty())
		devtype.push_back("");
// 	cout << "Spec size: " << devlist.size() << endl;
// 	cout << "Opt size: " << devtype.size() << endl;
	while (!result) {
		cout << endl;
		if (ispec < devlist.size()) {
			serdev = devlist[ispec++];
			opt = false;
		} else if (iopt < devtype.size() && nonexist) {
			serdev = devtype[iopt++];
			opt = true;
			nonexist = false;
			ocount = 0;
		} else if (iopt >= devtype.size()) {
			cout << "Could not find a compatible radio. If you know which device your\n";
			cout << "radio is attached to, then edit the config file to specify it.\n";
			cout << "If your radio could be on several ports, read the instructions in\n";
			cout << "the config file to enter the proper values.\n\n";
			cout << "HD Radio Controller will not work.\n";
			return false;
		}
		testdev = serdev;
		if (opt) {
			stringstream xfer;
			xfer << ocount++;
			testdev += xfer.str();
		}
// 		cout << "Result: " << result << ", Opt: " << opt << ", ISpec: " << ispec << ", IOpt: " << iopt << ", Device: " << testdev << endl;
		fstat = stat(testdev.c_str(), &finfo);
		if (fstat == 0) {
			result = testport(testdev);
		} else {
			nonexist = true;
		}
		if (ocount > 10) {exit(1);}
	}
	return true;
}

/**
 * Set the serial port attributes so we can communicate with the radio.  This
 * could effect the state of the DTR line in some cases.
 * @param portfd the file descriptor for the port we're setting
 */
void LinuxPort::setportattr(int portfd) { //public
// 	tcgetattr(portfd, &options);
//Set the baud rate (duh!)
//Should never change, set at 115200
	options.c_cflag = 0 | BAUD;
// 	options.c_cflag = BAUD;
//Next 4 set 8N1, 8 data bits, no parity, 1 stop bit
//DO NOT CHANGE
// 	options.c_cflag = CS8;
	options.c_cflag &= ~PARENB;
	options.c_cflag &= ~CSTOPB;
	options.c_cflag &= ~CSIZE;
	options.c_cflag |= CS8;
//Always use!
	options.c_cflag |= (CLOCAL | CREAD);
//disable RTS/CTS line for flow control
//works on RS232 and USB on original system
	options.c_cflag &= ~CRTSCTS;
//Needed to make sure the radio disconnects when this program
//exits or is killed or crashes or stops in some way that we
//can't control.
	options.c_cflag |= HUPCL;
//Enable software flow control
//Note relationship with xon/xoff software flow and RTSCTS
//hardware control.  If one's on, the other should be off.
//Works on RS232 and USB on original test system
// 	options.c_iflag = (IGNBRK | IGNPAR);
// 	options.c_iflag |= (IXON | IXOFF | IXANY);
	options.c_iflag = 0 | (IGNBRK | IGNPAR);
	options.c_iflag = (IXON | IXOFF | IXANY);
//Disable software flow control
//Works on RS232 but not on USB on original test system
//(Should not work, since software control is needed)
// 	options.c_iflag &= ~(IXON | IXOFF | IXANY);
//Suggested by Paul on testing for USB
	options.c_oflag = 0;
	options.c_lflag = 0;
//enableable RTS/CTS line for flow control
//horks system on RS232 and USB on original system
	int rc = tcsetattr(portfd, TCSANOW, &options);
	cout << "\tSerial port attribute set return code: " << rc << endl;
	return;
}

/**
 * We opne the serial port here.  The port should already be specified with setserialport().
 */
bool LinuxPort::openport() { //public
	if (isOpen) return true;
	isOpen = true;
	cout << "Opening port: " << serialdevice << endl;
	portfd = open(serialdevice.c_str(), O_RDWR | O_NOCTTY | O_NDELAY);
	setportattr(portfd);
	if (portfd == 0) {
		perror("\tUnable to open serial port.\n");
		return false;
	} else {
		cout << "\tPort " << serialdevice << " has been opened.  Descriptor: " << portfd << endl;
	}
	return true;
}

/**
 * Close the serial port for both input and output (the serial port is opened twice, once
 * for input and once for output.
 */
void LinuxPort::closeport() { //public
	isOpen = false;
	if (portfd == 0)
		return;
	if (verbose) printdtr("About to close ports");
	close(portfd);
	cout << "Serial port closed: " << serialdevice << endl;
	return;
}

/**
 * Set the serial port to use for communication.  When it's set here, it's also
 * set in the config so it'll be written out to the config file as the last known
 * port.  During autodiscovery that won't help, but the config file isn't saved during
 * autodiscovery.  It is saved whenever settings on the radio change, so setting the
 * serial port in the config class here will ensure the port is saved when the radio
 * is tuned to any station.
 *
 * On startup next time, the port saved will be used again unless overridden.
 * @param sport serial port
 */
void LinuxPort::setserialport(string sport) { //public
	serialdevice = sport;
	if (verbose) cout << "Setting serial device: " << serialdevice << endl;
	mainConfig->setval("device", serialdevice);
	return;
}

/**
 * Get the currently used serial port file name.
 * @return the path or name of the currently used serial device
 */
string LinuxPort::getserialport() { //public
	return serialdevice;
}

/**
 * Send a byte to the radio through the serial port.  Actually just calls hdsendbytes() and says
 * to send only 1 byte.
 * @param outbyte character or byte to send through the port
 */
void LinuxPort::hdsendbyte(char outbyte) { //public
	write(portfd, &outbyte, 1);
// 	cout << "Sending byte...\n";
// 	int rc = write(portfd, &outbyte, 1);
// 	cout << "Send bytes, FD: " << portfd << ", Return code: " << rc << endl;
	return;
}

/**
 * Send a series of bytes or characters through the serial port to the device.
 * @param outbytes the characters/bytes to send
 */
void LinuxPort::hdsendbytes(char* outbytes) { //public
	write(portfd, outbytes, sizeof(outbytes));
// 	int rc = write(portfd, outbytes, sizeof(outbytes));
// 	cout << "Send bytes, FD: " << portfd << ", Return code: " << rc << endl;
	return;
}

/**
 * Read the specified number of bytes from the serial port.  Our buffer is 1k, since
 * we're never working with anywhere near that much data anyway.
 * @param ilen number of bytes to read in
 * @return pointer to the buffer where  returned data is stored.
 */
char* LinuxPort::hdreadbytes(int ilen) { //public
	char* buff = new char[1024];
	int rd = 0;
	bool loopflag = false;

	buff[0] = 0;
	if (ilen > 1024)
		return buff;
// 	cout << "Entering wait/read loop, FD: " << portfd << ", Length: " << ilen << endl;
	if (!isOpen) {
		cout << "Having to re-open port\n";
		openport();
	}
	while (rd < 1) {
// 		cout << "Waiting for byte...\n";
		rd = read(portfd, buff, ilen);
		usleep(naptime);
// 		if (!loopflag && rd < 1) cout << "\nWaiting, read length: " << rd << endl;
// 		if (!loopflag) cout << "\tLast read: " << lastread << ", Waiting...\n";
		loopflag = true;
	}
	lastread = rd;
// 	cout << "Got a byte, read length: " << rd << endl;
	return buff;
}

/**
 * Read a single byte into a buffer and return it.
 * @return pointer to a 1 byte buffer with one byte read from the port
 */
char* LinuxPort::hdreadbyte() { //public
	char* buff = hdreadbytes(1);
	return buff;
}

/**
 * Number of bytes read in on the last read.
 * @return number of bytes read in
 */
int LinuxPort::hdlastreadleangth() { //public
	return lastread;
}

/**
 * Add a device to check when searching for the radio in autodiscovery.  All devices
 * given will be searched in order if autodiscovery is used to find the radio.
 * @param devname file path for the device to add
 */
void LinuxPort::addsearchdevice(string devname) { //public
	devlist.push_back(devname);
	return;
}

/**
 * Add a "type" of device to search for the radio.  This is mainly used to allow multiple
 * ports with almost the same filenames to be searched.  For example, specifying "/dev/ttyS"
 * instead of "/dev/ttyS0" will search multiple serial ports.  It'll start with adding a "0" on
 * the end ("/dev/ttyS0") then a "1" ("/dev/ttyS1") and so on.  It stops the first time a device
 * to check does not exist, then moves on to the next type.  This can be helpful on systems with
 * a number of serial port adaptors on USB connetions, since specifying "/dev/ttyUSB" will
 * automatically check all the devices starting with "/dev/ttyUSB0" and continue to the first
 * non-existant device.  Device types are searched *after* devices.
 * @param devicetype device type to search.
 */
void LinuxPort::addsearchtype(string devicetype) { //public
	devtype.push_back(devicetype);
	return;
}

/**
 * Simple routine to output the character in a readable 0x00 format,
 * followed by an int version, followed by a printable character (if
 * it is printable).  There is one subtlity: if we have a code for the
 * current characters (as in we know we're getting specific data), then
 * the separator between the hex and dec numbers is ":", if we don't have
 * a code, it's "-".  The A4 code saying we're starting data will always
 * have a "-" in it.
 * @param cIn the character to print
 */
void LinuxPort::chout(unsigned char cIn) {//protected

	char chex [10];
	int iIn;

	iIn = cIn;
	if (iIn == 0xA4) {
		cout << endl << endl;
	}
	sprintf(chex, "0x%2X", cIn);
	if (chex[2] == ' ')
		chex[2] = '0';
	cout << chex << ":" << iIn << " ";
// 	cout << endl;
	return;
}

/**
 * Another debugging routine.  Call with true to turn DTR on, false to turn it off.
 * @param newstate to set DTR to
 */
void LinuxPort::toggledtr(bool newstate) {//public
	int status;
	if (verbose) printdtr("---->Before toggling DTR");
	ioctl(portfd, TIOCMGET, &status);
	if (newstate) {
		cout << "Setting dtr\n";
		status |=  TIOCM_DTR;
	} else {
		cout << "Clearing dtr\n";
		status &= ~TIOCM_DTR;
	}
	ioctl(portfd, TIOCMSET, &status);
	if (verbose) printdtr("---->After toggling DTR");

	return;
}

/**
 * Overload with no argument for debugging DTR issues.
 */
void LinuxPort::printdtr() {//public
	printdtr("");
	return;
}

/**
 * Mainly for debugging, this will print out the state of the DTR line.
 */
void LinuxPort::printdtr(string msg) {//public
	bool dtrstate;
	dtrstate = getdtr();
// 	int status;
// 	fcntl(portfd, F_SETFL, 0);
// 	ioctl(portfd, TIOCMGET, &status);
	cout << msg << ", DTR State: " << dtrstate << endl;
	return;
}

/**
 * Mainly for debugging, get DTR state.
 * @return true for high, false for low.
 */
bool LinuxPort::getdtr() {//public
	bool result = false;
	int status;
// 	fcntl(portfd, F_SETFL, 0);
	ioctl(portfd, TIOCMGET, &status);
	if (status & TIOCM_DTR == 1)
		result = true;
	return result;
}

/**
 * Normally we do hang up, or turn off the radio, on exit, but in
 * some systems it may be desireable to leave the radio on on exit, so
 * that can be done by calling here to override the default with "false".
 * @param hangup true to turn off the radio on exit, false to leave it on
 */
void LinuxPort::hanguponexit(bool hangup) {//public
	tcgetattr(portfd, &options);
	if (hangup)
		options.c_cflag |= HUPCL;
	else
		options.c_cflag &= ~HUPCL;
	tcsetattr(portfd, TCSANOW, &options);
	return;
}

