package com.hal.gui;
/**
Program: TNet Client Letter Printer
Classes: HelpPanel
Author : Hal Vaughan
Company: Threshold Digital, Ltd.
			4221 Wakefield Rd.
			Richmond, VA 23235
			(804)-560-2820
			thresholddigital.com
			hal@thresholddigital
Purpose: This is part of the set of classes that goes on a client system.
TNet is part of a system to prepare data for clients, which is then
custom printed in the client's office in formats they specify.  This
system retreives data via e-mail, decrypts it and verifies the signature,
then unzips archive files and prints out all files sent.  (Batch files
can also be sent to change settings.)  When done, we send e-mail back to
report to the server.  All printing and formatting is done with OpenOffice.org,
which is open source and available at openoffice.org.  We use this program
because it is free (as in speech and as in beer) and the data file formats are
clearly documented for us to use and modify (unlike certain other office suites).

Specifics:
	HelpPanel: Actually includes tPanel (It was 2:30 am -- easier to cut and paste
	than to think!).  This creates a panel with a lower etched border and a title
	and loads in the specified html help file into a non-editable JTextEditor panel
	with a scrollbar.
*/

import java.awt.Color;
import java.awt.Dimension;
import java.net.URL;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.border.Border;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

/**
 * Create a simple HTML panel that can have a border and title or not.  It displays plain
 * text or HTML formatted text.
 * @author hal
 *
 */
public class HTMLPanel extends JPanel {

	private static final long serialVersionUID = 1L;
	
	private boolean useEtched = true;
	
	private JEditorPane pHelp;
	
	private Border loweredEtched;
	
	private TitledBorder tBorder;
	
	private String panelTitle = "";

	/**
	 * Create a new HTML panel
	 * @param i1 width of panel
	 * @param i2 height of panel
	 * @param displayText text to display
	 * @param panelTitle title for etched border
	 */
	public HTMLPanel(int i1, int i2, String displayText, String panelTitle) {
		makePanel(new Dimension(i1, i2), displayText, panelTitle);
		return;
	}

	/**
	 * Make an HTML panel
	 * @param d1 dimension or size of panel
	 * @param displayText text to display
	 * @param panelTitle title to show in border
	 */
	public HTMLPanel(Dimension d1, String displayText, String panelTitle) {
		makePanel(d1, displayText, panelTitle);
		return;
	}

	/**
	 * Create an HTML panel
	 * @param displayText text to display
	 * @param panelTitle title of panel to display in border
	 */
	public HTMLPanel(String displayText, String panelTitle) {
		makePanel(new Dimension(0, 0), displayText, panelTitle);
		return;
	}

	/**
	 * Internal stuff for both constructors
	 * @param pSize size of panel, or 0,0 if we don't specify a size
	 * @param displayText text to display
	 * @param sTitle border title
	 */
	private void makePanel(Dimension pSize, String displayText, String sTitle) {
		String sMode = "";
		URL oURL = null;
		panelTitle = sTitle;
		createBorder();
		setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
		pHelp = new JEditorPane();
		JScrollPane pHelpScroll = new JScrollPane(pHelp);
		pHelpScroll.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		setText(displayText);
		try {
			oURL = new URL(sMode);
			pHelp.setPage(oURL);
		} catch (Exception e) {}
		pHelp.setEditable(false);
		pHelp.setBackground(Color.LIGHT_GRAY);
		add(pHelpScroll);
		if ((pSize.getHeight() > 0) && (pSize.getWidth() > 0)) {
			setMaximumSize(pSize);
			setMinimumSize(pSize);
			setPreferredSize(pSize);
		}
		return;
	}
	
	/**
	 * Create a border to use with the specified title.
	 * @param borderTitle title to use for etched border.
	 */
	private void createBorder() {
		if (useEtched) {
			loweredEtched = BorderFactory.createEtchedBorder(EtchedBorder.LOWERED);
			tBorder = BorderFactory.createTitledBorder(loweredEtched, panelTitle);
			tBorder.setTitleJustification(TitledBorder.DEFAULT_JUSTIFICATION);
			tBorder.setTitlePosition(TitledBorder.DEFAULT_POSITION);
			setBorder(tBorder);
		} else {
			setBorder(null);
		}
		return;
	}
	
	/**
	 * Set the title, which means making up a new border.  Automatically creates and
	 * uses the etched border as well.
	 * @param newTitle new title
	 */
	public void setTitle(String newTitle) {
		panelTitle = newTitle;
		createBorder();
		return;
	}
	
	/**
	 * Set the text for display (HTML will NOT be parsed!).
	 * @param displayText text to display
	 */
	public void setText(String displayText) {
		pHelp.setText(displayText);
		return;
	}
	
	/**
	 * Specify a file to use as the source for text in the panel.
	 * @param sourceFile file with text to display
	 */
	public void setFile(String sourceFile) {
		try {
			URL fileURL = new URL("file:" + sourceFile);
			pHelp.setPage(fileURL);
		} catch (Exception e) {}
		return;
	}
	
	/**
	 * Specify whether to use an etched border with a title around the panel.
	 * @param useEtched true to use border
	 */
	public void useEtchedBorder(boolean useEtched) {
		this.useEtched = useEtched;
		createBorder();
		return;
	}

}
