package com.hal.gui;

import java.awt.Color;
import java.awt.Component;
import java.util.LinkedList;

import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * This is intended to allow special ways to render cells in tables.  It was originally written
 * to simply allow certain rows of a table to be rendered in a different color if that row was in
 * a list of rows.  It is designed so other terms and conditions can be added along the way, such
 * as specifying that certain columns be rendered, or columns and rows, or cells meeting certain
 * conditions.<br><br>
 * 
 * To add, use JTable.addDefaultCellRenderer(Object.class, new HalCellRenderer()).<br><br>
 * 
 * The only problem, though, with adding it as a new object in itself is that it makes it impossible
 * to make changes to the rendering conditions without getting the renderer from the table.
 * @author hal
 */
public class HalCellRenderer extends DefaultTableCellRenderer {
//public class HalCellRenderer {
	
	private static final long serialVersionUID = 1L;
	
	boolean colorByRow = false;
	
	int cellRow, cellCol;
	LinkedList specialRows;
	
	Component cellRenderer;
	
	Color defaultBackgroundColor = Color.WHITE, specialBackgroundColor = Color.WHITE;
	
	JTable sourceTable;
	
	/**
	 * Simple constructor.  Nothing outstanding.  Just initialize it all.
	 */
	public HalCellRenderer() {
		super();
		specialRows = new LinkedList();
	}
	
	/**
	 * Set to color items a special color (which can be set) if they
	 * are included in a list of rows given to us.
	 * @param colorByRow
	 */
	public void colorByRow(boolean colorByRow) {
		this.colorByRow = colorByRow;
		return;
	}
	
	/**
	 * Used by the table to do the rendering.  We'll have to add more conditions to this as we add more possibilities for the renderer.
	 */
	public Component getTableCellRendererComponent(JTable sourceTable, Object cellValue, boolean isSelected, boolean hasFocus, int cellRow, int cellCol) {
		Integer iVal;
		cellRenderer = super.getTableCellRendererComponent(sourceTable, cellValue, isSelected, hasFocus, cellRow, cellCol);
//		System.out.println("Rednering cell, Row: " + cellRow + ", Col: " + cellCol);
		if (colorByRow) {
//			System.out.println("\tColoring by row.");
			iVal = Integer.valueOf(cellRow);
			if (specialRows.contains(iVal)) {
//				System.out.println("\t\tUsing special coloring.");
				cellRenderer.setBackground(specialBackgroundColor);
			} else {
				cellRenderer.setBackground(defaultBackgroundColor);
			}
				
		}
		return cellRenderer;
	}
	
	/**
	 * Set the default background color.  It defaults to white.
	 * @param defColor the color background cells will be if they don't meet special conditions.
	 */
	public void setDefaultBackgroundColor(Color defColor) {
		defaultBackgroundColor = defColor;
		return;
	}
	
	/**
	 * Set the background color to use if a cell meets special conditions for coloring.
	 * @param specialColor color to use for special rendering
	 */
	public void setSpecialBackgroundColor(Color specialColor) {
		specialBackgroundColor = specialColor;
		return;
	}
	
	/**
	 * Add a set of row numbers that are supposed to be colored with a special color.
	 * @param iRows rows to color
	 */
	public void addSpecialRows(int[] iRows) {
		int x;
		for (x = 0; x < iRows.length; x++) {
			addSpecialRow(iRows[x]);
		}
		return;
	}
	
	/**
	 * Add one row that is supposed to be colored with a special color.
	 * @param iRow single row to add for coloring
	 */
	public void addSpecialRow(int iRow) {
		specialRows.add(Integer.valueOf(iRow));
		return;
	}
	
	/**
	 * Clear the list of special rows for special coloring.
	 */
	public void resetSpecialRows() {
		specialRows = new LinkedList();
		return;
	}
	

}
