package com.hal.gui;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
/**
 * Provide a simple progress meter.  Basically make the decisions here so it's easy to 
 * use progress meters.  We make it, set the task length, then check regularly for
 * a cancel and update it with a new number showing how far along we are when possible.
 * When done we close it.
 * @author hal
 *
 */
public class HalProgressMonitor implements ActionListener {

	private static final long serialVersionUID = 1L;
	
	static LinkedList allMonitors = new LinkedList();
	
	public boolean isCancelled = false;
	
	JLabel jMessage;
	
	JButton jbCancel;
	
	JProgressBar jProgress;
	
	public JFrame pFrame = new JFrame();
	JFrame parentFrame = null;
	
	/**
	 * Simple way to create it.  Just give us a parent frame and whatever message to
	 * display on the window
	 * @param parentFrame parent frame
	 * @param sMessage message to appear on the progress meter
	 */
	public HalProgressMonitor(JFrame parentFrame, String sMessage) {
		this.parentFrame = parentFrame;
		makeIt(sMessage);
	}
	
	/**
	 * No parent frame constructor.
	 * @param sMessage message to appear on progress meter
	 */
	public HalProgressMonitor(String sMessage) {
		makeIt(sMessage);
	}
	
	/**
	 * All the things both constructors use to create the meter.
	 * @param sMessage message to appear on progress meter
	 */
	private void makeIt(String sMessage) {
		isCancelled = false;
		pFrame.setPreferredSize(new Dimension(275, 125));
		pFrame.setMinimumSize(new Dimension(275, 125));
		pFrame.setSize(new Dimension(275, 125));
		pFrame.setTitle(sMessage);
		Container cPane = pFrame.getContentPane(); 
		cPane.setLayout(new BoxLayout(cPane, BoxLayout.Y_AXIS));
		jMessage = new JLabel(sMessage);
		jProgress = new JProgressBar();
		jProgress.setIndeterminate(true);
		jProgress.setStringPainted(true);
		jProgress.setString("Waiting...");
		jbCancel = new JButton("Cancel");
		jbCancel.addActionListener(this);
		JPanel topPanel = new JPanel();
		JPanel midPanel = new JPanel();
		JPanel lowPanel = new JPanel();
		topPanel.setLayout(new BoxLayout(topPanel, BoxLayout.X_AXIS));
		midPanel.setLayout(new BoxLayout(midPanel, BoxLayout.X_AXIS));
		lowPanel.setLayout(new BoxLayout(lowPanel, BoxLayout.X_AXIS));
		topPanel.add(Box.createHorizontalGlue());
		topPanel.add(jMessage);
		topPanel.add(Box.createHorizontalGlue());
		midPanel.add(Box.createHorizontalGlue());
		midPanel.add(jProgress);
		midPanel.add(Box.createHorizontalGlue());
		lowPanel.add(Box.createHorizontalGlue());
		lowPanel.add(jbCancel);
		lowPanel.add(Box.createHorizontalGlue());
		cPane.add(Box.createVerticalGlue());
		cPane.add(topPanel);
		cPane.add(Box.createVerticalGlue());
//		cPane.add(Box.createRigidArea(new Dimension(10, 4)));
		cPane.add(midPanel);
		cPane.add(Box.createVerticalGlue());
//		cPane.add(Box.createRigidArea(new Dimension(10, 4)));
		cPane.add(lowPanel);
		cPane.add(Box.createVerticalGlue());
//		this.pack();
		positionWindow();
//		allMonitors.add(this);
//		pFrame.setVisible(true);
		return;
	}
	
	public void activate(String sMessage) {
		positionWindow();
		reset(sMessage);
		pFrame.setVisible(true);
		allMonitors.add(this);
		return;
	}
	
	public void activate() {
		reset();
		pFrame.setVisible(true);
		allMonitors.add(this);
		return;
	}
	
	/**
	 * Close the progress meter.
	 */
	public void deactivate() {
		pFrame.setVisible(false);
		allMonitors.remove(this);
	}
	
	/**
	 * Start over, but change the message.  Don't close, just do a Cylon thing
	 * until we get new task info.
	 * @param newMessage
	 */
	public void reset(String newMessage) {
		pFrame.setTitle(newMessage);
		jMessage.setText(newMessage);
		reset();
		return;
	}
	
	/**
	 * Start over.  Don't close, just do the Cylon thing until we get
	 * new task info.
	 */
	public void reset() {
		jProgress.setIndeterminate(true);	
		jProgress.setString("Waiting...");
		isCancelled = false;
		return;
	}
	
	/**
	 * Set the text and title in the window
	 * @param newTitle the new window title
	 * @param newMessage the new message for hte user
	 */
	public void setData(String newTitle, String newMessage) {
		pFrame.setTitle(newTitle);
		jMessage.setText(newMessage);
		return;
	}
	
	/**
	 * Move all progress monitors to the front of the screen, starting with the first one.
	 * Many times a dialog or other window can move the parent window in front of other
	 * windows, like the progress monitors, so this will move them all to the front.
	 *
	 */
	public void moveAllToFront() {
		Iterator iPoint = allMonitors.iterator();
		HalProgressMonitor hpMon;
		while (iPoint.hasNext()) {
			hpMon = (HalProgressMonitor) iPoint.next();
			hpMon.pFrame.toFront();
		}
		return;
	}
	
	/**
	 * Specify a start number and end number to define the task length.  Until
	 * this is set, the meter stays in Cylon mode.  Once specified, it goes to
	 * a normal mode.
	 * @param iStart start of task (usually 0)
	 * @param iEnd end of task
	 */
	public void setTaskLength(int iStart, int iEnd) {
		jProgress.setIndeterminate(false);
		jProgress.setMinimum(iStart);
		jProgress.setMaximum(iEnd);
		jProgress.setValue(0);
		jProgress.setString(null);
		return;
	}
	
	/**
	 * Update the progress bar with a number between the start and end of the task
	 * to specify how far along we are in the task.
	 * @param iPosition current position in task completion
	 */
	public void updateProgressBar(int iPosition) {
		jProgress.setValue(iPosition);
		return;
	}
	
	/**
	 * Position the window.  Since we have cases with multiple levels of tasks, there
	 * could be more than one task being monitored at a time, so after the first one is
	 * positioned, others are below it (or above if it was at the bottom of the screen).
	 * This puts the window in position relative to a parent frame or to othe progress
	 * monitors.
	 */
	public void positionWindow() {
		int xPos = 0, yPos = 0, yLast = 0, ourHeight = 0, ourWidth = 0;
		Point pCurrent = null;
		Dimension dScreen = Toolkit.getDefaultToolkit().getScreenSize();
		Dimension ourSize = pFrame.getSize();
		ourHeight = ourSize.height;
		ourWidth = ourSize.width;
		yLast = dScreen.height;
		HalProgressMonitor lastMonitor = null;
		try {
			lastMonitor = (HalProgressMonitor) allMonitors.getLast();
			pCurrent = lastMonitor.pFrame.getLocationOnScreen();
		} catch (Exception e) {}
		if (pCurrent != null) {
			xPos = pCurrent.x;
			yPos = pCurrent.y;
			yPos = yPos + lastMonitor.pFrame.getHeight() + 20;
		} else if (parentFrame != null){
			pCurrent = parentFrame.getLocation();
			Dimension dCurrent = parentFrame.getSize();
			xPos = pCurrent.x;
			yPos = pCurrent.y;			
			xPos = xPos + ((dCurrent.width - ourSize.width) / 2);
		} else {
			xPos = (dScreen.width - ourWidth) / 2;
			yPos = dScreen.height / 4;
		}
		if (yPos >= (yLast - ourHeight)) {
			yPos = 0;
		}
		pFrame.setLocation(xPos, yPos);
		return;
	}
	
	/**
	 * Listen for cancellation.  If so, set isCancelled to true.
	 */
	public void actionPerformed(ActionEvent aEvent) {
		Object sourceControl = aEvent.getSource();
		if (sourceControl == jbCancel) {
			isCancelled = true;
		}
		return;
	}
}
