package com.hal.networking;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.URL;
import java.net.URLConnection;

import javax.swing.JFrame;

import com.hal.gui.HalProgressMonitor;
import com.hal.util.HalFile;
import com.hal.util.StringHashMap;

public class WebIO {
	
	byte[] crossThreadByte = null;
	
	public boolean appendData = false, showProgress = false, crossThreadBoolean, isAborted = false;
	
	private String myURL = "", webPage = "";
	
	private String tempDir;
	
	String idString = "HalLibWebIO.img";
	
	private StringHashMap formData = new StringHashMap();
	
	HalFile hFile;
		
	HalProgressMonitor pMonitor = null;
	
	JFrame parentFrame = null;
	
	private StringHashMap statAbort, statError, statLength, statRead;
	
	/**
	 * Default simplified constructor.  Move along.  Nothing to see here.
	 */
	public WebIO() {
		statAbort = new StringHashMap();
		statError = new StringHashMap();
		statLength = new StringHashMap();
		statRead = new StringHashMap();
		hFile = new HalFile();
		tempDir = System.getProperty("java.io.tmpdir") + File.separator;
	}
	
	/**
	 * Set the parent frame for GUI programs.  Sometims we're used from the command line
	 * and sometimes by a GUI program.  A GUI program can set the parentFrame and will then
	 * get a progress bar if requested before doing certain operations.
	 * @param parentFrame the frame all progress bars will be aligned with
	 */
	public void setParentFrame(JFrame parentFrame) {
		this.parentFrame = parentFrame;
		return;
	}
	
	/**
	 * Specify whether a visibile progress bar should be shown during some activities.
	 * @param bProgress true to show progress bar, false to no.  Default is false.
	 */
	public void setVisibleProgress(boolean bProgress) {
		showProgress = bProgress;
		return;
	}
	
	/**
	 * Set the default URL value
	 * @param newURL value for the new URL.
	 */
	public void setURL(String newURL) {
		myURL = newURL;
		return;
	}
	
	/**
	 * Add a field/value pair to those used if we're going to POST form data.
	 * @param fieldName name the field we're adding
	 * @param fieldValue value of the field we're adding
	 */
	public void addFormValue(String fieldName, String fieldValue) {
		formData.put(fieldName, fieldValue);
		return;
	}
	
	/**
	 * Clear all the form field/value pairs.
	 */
	public void clearFormData() {
		formData = new StringHashMap();
		return;
	}
	
	/**
	 * Find out if an operation was aborted or not
	 * @param statText text id of the read operation
	 * @return true if it was intentionally aborted
	 */
	public boolean getAbortStatus(String statText) {
		boolean abortStatus = false;
		String sStat = statAbort.get(statText);
		if (sStat.equals("true"))
			abortStatus = true;
		return abortStatus;
	}
	
	/**
	 * Find out if an operation had an error
	 * @param statText
	 * @return true if there were errors reading the URL or saving data to a file
	 */
	public boolean getErrorStatus(String statText) {
		boolean errorStatus = true;
		if (statError.get(statText).equals(""))
			errorStatus = false;
		return errorStatus;
	}
	
	/**
	 * Get the error message if an operation had an error.  If no error, an empty
	 * string will be returned.
	 * @param statText
	 * @return true if there were errors reading the URL or saving data to a file
	 */
	public String getErrorMessage(String statText) {
		String errorMsg = statError.get(statText);
		if (errorMsg.equals("false"))
			errorMsg = "";
		return errorMsg;
	}
	
	/**
	 * Get the total length of the file or page we downloaded
	 * @param statText text id of the read operation
	 * @return total length of download (as given by server, not how much we read)
	 */
	public String getFileLength(String statText) {
		String outFile = statLength.get(statText);
		return outFile;
	}
	
	/**
	 * Find out how much data was read in an operation
	 * @param statText text id of the read operation
	 * @return bytes read
	 */
	public String getReadLength(String statText) {
		String outFile = statRead.get(statText);
		return outFile;
	}

	/**
	 * Get the data from a specified operation instead of just the file name.
	 * @param statText text id of the read operation
	 * @return file data from download
	 */
	public byte[] getOutputData(String statText) {
		int iLen = 0;
		byte[] bNull = new byte[0];
		File outFile = new File(statText);
		
//		System.out.println("DEBUG: Reading file: " + sName);
		if (!outFile.exists()) {return bNull;}
		FileInputStream fis = null;
		iLen = (int) outFile.length();
		byte[] outData = new byte[iLen];
		try {
			fis = new FileInputStream(statText);
		} catch (FileNotFoundException e) {
			System.out.println("A file that was there is gone!  No tmp file in WebIO: ");
			e.printStackTrace();
			return ("Error: File not found").getBytes();
		}
		DataInputStream dis = new DataInputStream(fis);
		try {
			dis.read(outData, 0, iLen);
			dis.close();
			fis.close();
		} catch (IOException e) {
			return ("Error: IO Exception").getBytes();
		}
		return outData;
	}

	/**
	 * Get the text data from a specified operation instead of just the file name.
	 * @param statText text id of the read operation
	 * @return text file data from download
	 */
	public String getOutputText(String statText) {
		byte[] binPage;
		String textPage = "";
		binPage = getOutputData(statText);
		textPage = new String(binPage);
		return textPage;
	}
	
	/**
	 * Once we've gotten all the data from a download, we can trash the
	 * hash data associated with it AND delete the temp file created to store the data.
	 * @param statText statText text id of the read operation
	 */
	public void resolveAction(String statText) {
		File tFile = new File(statText);
		statAbort.remove(statText);
		statError.remove(statText);
		if (tFile.exists())
			tFile.delete();
		if (tFile.exists())
			tFile.deleteOnExit();
		return;
	}
	
	/**
	 * Get the web page at the current default URL.
	 * @return text data from page at URL.
	 */
	public String getURL() {
		webPage = getURL(myURL);
		return webPage;
	}
	
	/**
	 * Get the web page at the url we specify.  Do NOT post any
	 * form data.
	 * @param pageURL  The URL to open and get data from.
	 * @return text version of data at the specified URL.
	 */
	public String getURL(String pageURL) {
		webPage = getURL(pageURL, false);
		return webPage;
	}
	
	/**
	 * Open the default location as a URL, POST any existing form data to it,
	 * then get the web page that's returned. 
	 * @return text form of returned web page
	 */
	public String getFormURL() {
		String statCode = getURL(myURL, true);
		return statCode;
	}
	
	/**
	 * POST our form data to the URL we provide and get the text version
	 * of the web page that is returned from the server.
	 * @param formURL text form of the URL we're POSTing to
	 * @return text form of the web page that is returned from our POST
	 */
	public String getFormURL(String formURL) {
		webPage = getURL(formURL, true);
		return webPage;
	}
	
	/**
	 * Do the actual work for specifying a URL, opening it, POSTing form data
	 * (if specified to do so), and getting the data from the form.  This routine actually
	 * callse the other one that does the work.  This is for convenience for calls not
	 * including a progress monitor.
	 * @param pageURL URL to use
	 * @param postData true if we POST form data, false if we don't
	 * @return text version of data at page located at URL
	 */
	public String getURL(String pageURL, boolean postData) {
		String statText = getURL(pageURL, postData, null);
		return statText;
	}
	
	/**
	 * Do the actual work for specifying a URL, opening it, POSTing form data
	 * (if specified to do so), and getting the data from the form.
	 * @param pageURL URL to use
	 * @param postData true if we POST form data, false if we don't
	 * @return text version of data at page located at URL
	 */
	public String getURL(String pageURL, boolean postData, HalProgressMonitor hpMonitor) {
		char[] dlData = null;
		boolean bAbort = false, deleteTempFile = false;
		int x, iLong = -1, iRead = 0, iLimit = 4096, iTotal = 0;
		String aID = HalFile.makeName(tempDir, idString), sLine;
		URLConnection ucPage = null;
		PrintWriter outPrint;
		BufferedReader inRead;
		BufferedWriter dataOut;
		
		statRead.put(aID, iTotal);
		statLength.put(aID, iLong);
		statAbort.put(aID, String.valueOf(bAbort));
		statError.put(aID, "");
		
		File oFile = new File(aID);
		try {
			oFile.createNewFile();
			oFile.deleteOnExit();
		} catch (Exception e) {
			statError.put(aID, String.valueOf("Could not create temporary data storage file."));
			return aID;
		}
		String[] formKeys = formData.keySet();
		
		sLine = "";
		for (x = 0; x < formKeys.length; x++) {
			sLine = sLine + formKeys[x] + "=" + formData.get(formKeys[x]);
			if (x < formKeys.length - 1) {
				sLine = sLine + "&";
			}
		}
		if (postData && appendData) {
//			System.out.println("Appending data for POST.");
			pageURL = pageURL + "?" + sLine;
//			System.out.println("\tFull URL: " + pageURL);
		}
//		System.out.println("Full URL: " + pageURL);
//		System.out.println("POST info: " + sLine);
//		System.out.println("POST: " + postData + ", Append: " + appendData);
		
		try {
			ucPage = (new URL(pageURL)).openConnection();
			ucPage.setDoOutput(postData && !appendData);
			ucPage.setDoInput(true);
			if (postData && !appendData) {
//				System.out.println("POSTing without appending");
				outPrint = new PrintWriter(ucPage.getOutputStream());
				outPrint.print(sLine);
				outPrint.close();
				outPrint.flush();
			}
			iLong = ucPage.getContentLength();
			inRead = new BufferedReader(new InputStreamReader(ucPage.getInputStream()));
			dataOut = new BufferedWriter(new FileWriter(aID)); 
			if (hpMonitor != null && iLong > 0) hpMonitor.setTaskLength(0, iLong);
			dlData = new char[iLimit];
//			System.out.println("Total page length: " + iLong);
			iRead = inRead.read(dlData, 0, iLimit);
			while (iRead != -1) {
				if (hpMonitor != null) {
					isAborted = hpMonitor.isCancelled;
					if (isAborted) {
						statAbort.put(aID, String.valueOf(isAborted));
						deleteTempFile = true;
						break;
					}
				}
				dataOut.write(dlData, 0, iRead);
				iTotal = iTotal + iRead;
				if (hpMonitor != null) {
					hpMonitor.updateProgressBar(iTotal);
				}
				statRead.put(aID, iTotal);
				iRead = inRead.read(dlData, 0, iLimit);
			}
//			System.out.println("Data length (read in): " + iTotal);
			inRead.close();
			dataOut.close();
			if (deleteTempFile) {
				if (oFile.exists())
					oFile.delete();
				if (oFile.exists())
					oFile.deleteOnExit();
			}
		} catch (Exception e) {
			statError.put(aID, "Error in reading data from web or writing it to disk.  Exception: " + e.toString());
		}
//		System.out.println("File is all in place!");
		oFile.deleteOnExit();
		return aID;
	}
	
	/**
	 * Get binary data from a website (this could be text as well).  We don't return
	 * the data but the success code.  (false is error).  Once called, all the stats
	 * and data are available through various getXXXX() functions in this group.
	 * @param dataURL url to download
	 * @param postData true if we use form data already set for POST
	 * @return Action id info used to get the file, data, or status info on the download
	 */
	public String getBinaryURL(String dataURL, boolean postData) {
		String sResult = getBinaryURL(dataURL, postData, null);
		return sResult;
	}
	
	/**
	 * Get a binary file from a website (we could get text as well).  We don't return data,
	 * but a success code.  If this is called by a GUI program that needs a progress monitor
	 * update, then include the progress monitor as a parameter, otherwise make that one parameter
	 * null.
	 * @param dataURL url to download from
	 * @param postData true if we POST form data already set up
	 * @param hpMonitor GUI monitor to update
	 * @return statcode for operation
	 */
	public String getBinaryURL(String dataURL, boolean postData, HalProgressMonitor hpMonitor) {
		byte[] dlData = null;
		boolean bAbort = false, deleteTempFile = false;
		int x, iLong = -1, iRead, iLimit = 4096, iTotal = 0;
		String aID = HalFile.makeName(tempDir, idString), sLine;
		URLConnection ucPage = null;
		PrintWriter outPrint;
		BufferedInputStream inRead;
		FileOutputStream fileOut;
		BufferedOutputStream dataOut;
		
		statRead.put(aID, iTotal);
		statLength.put(aID, iLong);
		statAbort.put(aID, String.valueOf(bAbort));
		statError.put(aID, "");
		
		File oFile = new File(aID);
		try {
			oFile.createNewFile();
			oFile.deleteOnExit();
		} catch (Exception e) {
			statError.put(aID, String.valueOf("Could not create temporary data storage file."));
			return aID;
		}
		String[] formKeys = formData.keySet();
		
		sLine = "";
		for (x = 0; x < formKeys.length; x++) {
			sLine = sLine + formKeys[x] + "=" + formData.get(formKeys[x]);
			if (x < formKeys.length - 1) {
				sLine = sLine + "&";
			}
		}
		if (postData && appendData) {
//			System.out.println("Appending data for POST.");
			dataURL = dataURL + "?" + sLine;
//			System.out.println("\tFull URL: " + pageURL);
		}
//		System.out.println("Full URL: " + dataURL);
//		System.out.println("POST info: " + sLine);
//		System.out.println("POST: " + postData + ", Append: " + appendData);
		
		try {
			ucPage = (new URL(dataURL)).openConnection();
			ucPage.setDoOutput(postData && !appendData);
//			ucPage.setDoOutput(!appendData);
			ucPage.setDoInput(true);
			if (postData && !appendData) {
//				System.out.println("POSTing without appending");
				outPrint = new PrintWriter(ucPage.getOutputStream());
				outPrint.print(sLine);
				outPrint.close();
				outPrint.flush();
			}
			iLong = ucPage.getContentLength();
//			System.out.println("On first check, length: " + iLong);
			fileOut = new FileOutputStream(oFile);
			dataOut = new BufferedOutputStream(fileOut);
			inRead = new BufferedInputStream(ucPage.getInputStream()); 
			statLength.put(aID, iLong);
//			System.out.println("Length of file: " + iLong);
			if (hpMonitor != null) hpMonitor.setTaskLength(0, iLong);
			dlData = new byte[iLimit];
			iRead = inRead.read(dlData, 0, iLimit);
			while (iRead != -1) {
//Check for abort flag
				if (hpMonitor != null) {
					isAborted = hpMonitor.isCancelled;
					if (isAborted) {
						statAbort.put(aID, String.valueOf(isAborted));
						deleteTempFile = true;
						break;
					}
				}
//				dlData = new byte[iLimit];
				dataOut.write(dlData, 0, iRead);
				iTotal = iTotal + iRead;
				if (hpMonitor != null) {
					hpMonitor.updateProgressBar(iTotal);
				}
				statRead.put(aID, iTotal);
//				System.out.println("\tLoop.  Read this time: " + iRead + ", Total read: " + iTotal + ", Length: " + iLong);
				iRead = inRead.read(dlData, 0, iLimit);
			}
//			System.out.println("Total read in: " + iTotal + ", Reported length: " + iLong);
//			System.out.println("Done with outer loop!");
			inRead.close();
			dataOut.close();
			if (deleteTempFile) {
				if (oFile.exists())
					oFile.delete();
				if (oFile.exists())
					oFile.deleteOnExit();
			}
		} catch (Exception e) {
			statError.put(aID, "Error in reading data from web or writing it to disk.  Exception: " + e.toString());
		}
		oFile.deleteOnExit();
//		System.out.println("File is all in place!");
		return aID;
	}
	
}
