package com.hal.oldtimeradio;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import com.hal.conf.GeneralConfig;
import com.hal.gui.GUIUtils;
import com.hal.gui.HalCellRenderer;
import com.hal.gui.HelpWindow;
import com.hal.util.StringHashMap;

public class DownloadQueue implements ActionListener {

	private JFrame jFrame = null;  //  @jve:decl-index=0:visual-constraint="164,22"
	private JPanel jContentPane = null;
	private JLabel jLabel = null;
	private JLabel jlCount = null;
	private JScrollPane jScrollPane = null;
	private JTable jTable = null;  //  @jve:decl-index=0:visual-constraint="462,360"
	private JButton jbClear = null;
	private JButton jbRemove = null;
	private JButton jbHelp = null;
	private JButton jbDownload = null;
	private JPanel jButtons = null;
	private JPanel jPanel1 = null;
	private JButton jbClose = null;
	
	
	boolean isVisible = false, firstOpen = true, firstHelp = true;
	int lastDLCount = 0;
	String helpFile = "0000-OTRQueueHelp.html";
	String[] colNames = new String[] {"Show Title", "Episode Title", "URL", "Local File"};
	String[] hiddenColNames = new String[] {"URL", "Local File"};
	StringHashMap hiddenColumns;
	
	DefaultTableModel dataTable = null;
	
	OTRDownload otrControl;
	ShowData allShows;
	EpisodeData allEps;	
	GeneralConfig myConfig;

	HalCellRenderer hcRender;
	HelpWindow helpWindow;
	
	JFrame parentFrame = null;
	
	/**
	 * Simple, dull, boring constructor.  Nothing to see here.  Move along.
	 * @param programConfig basic eneral program config object
	 */
	public DownloadQueue(GeneralConfig programConfig) {
		int x;
		myConfig = programConfig;
		otrControl = (OTRDownload) myConfig.getObject("otrcontrol");
		allShows = (ShowData) myConfig.getObject("allshows");
		allEps = (EpisodeData) myConfig.getObject("allepisodes");
		hiddenColumns = new StringHashMap();
		for (x = 0; x < hiddenColNames.length; x++) {
			hiddenColumns.put(hiddenColNames[x], "true");
		}
		helpWindow = new HelpWindow(600, 400, "Help for download queue.", "Old Time Radio Show Downloader Title Help");
		helpFile = myConfig.get("OTRDirectory") + File.separator + helpFile;
		helpWindow.setFile(helpFile);
		guiSetup();		
		return;
	}
	
	/**
	 * Reset the queue to a length of zero.
	 */
	public void reset() {
		dataTable.setRowCount(0);
		setSizeDisplay();
		return;
	}
	
	/**
	 * Add a number of items when each episode is part of a different show.
	 * The show names in the first array have to be paired with the episode names
	 * in the second one.
	 * @param showNames list of shows to match the episodes to be added
	 * @param episodeFiles list of episode files to be added to the queue
	 */
	public void addItems(String[] showNames, String[] episodeFiles) {
		int x;
		for (x = 0; x < showNames.length; x++) {
			addItem(showNames[x], episodeFiles[x]);
		}
		return;
	}
	
	/**
	 * Overloaded to add a number of items to the queue when they all are episodes
	 * of the same show.  The show name only has to be givein once.
	 * @param showName name of show to add
	 * @param episodeFiles file names of all the episodes to add
	 */
	public void addItems(String showName, String[] episodeFiles) {
		int x;
		for (x = 0; x < episodeFiles.length; x++) {
			addItem(showName, episodeFiles[x]);
		}
		return;
	}
	
	/**
	 * Add an item to the end of the queue.  This can only be used if the main
	 * instance of AllEpisodes in the control panel is "matched" with the current
	 * show/episode set on display.
	 * @param episodeFile name of episode to add to queue
	 */
	public void addItem(String episodeFile) {
		String showName = allEps.getShow(episodeFile);
		addItem(showName, episodeFile);
		return;
	}
	
	/**
	 * Add an item to the end of the queue
	 * @param showName name of show to add to queue
	 * @param episodeFile file name of episode to add to queue
	 */
	public void addItem(String showName, String episodeFile) {
		String webURL = allEps.getURL(episodeFile);
		String localFile = allEps.getFilePath(episodeFile);
		dataTable.addRow(new String[] {showName, episodeFile, webURL, localFile});
		setSizeDisplay();
		return;
	}
	
	/**
	 * Get all the files to download in the queue.
	 * @return array of files to download, each item tab separated with the show title, a tab, and end title
	 */
	public String[] getQueue() {
		int x, iLimit;
		String showTitle, epFile, webURL, localFile;
		iLimit = dataTable.getRowCount();
		String[] allQueue = new String[iLimit];
//		System.out.println("Download Queue Length: " + iLimit);
		for (x = 0; x < iLimit; x++) {
			showTitle = (String) dataTable.getValueAt(x, 0);
			epFile = (String) dataTable.getValueAt(x, 1);
			webURL = (String) dataTable.getValueAt(x, 2);
			localFile = (String) dataTable.getValueAt(x, 3);
			allQueue[x] = showTitle + "\t" + epFile + "\t" + webURL + "\t" + localFile;
//			System.out.println("Q Line: " + allQueue[x]);
		}
		return allQueue;
	}
	
	/**
	 * Remove a number of rows from the queu
	 * @param iList number of each row to remove
	 */
	public void removeItems(int[] iList) {
		int x;
		for (x = iList.length -1; x >= 0; x--) {
			removeItem(iList[x]);
		}
		return;
	}
	
	/**
	 * Remove a single row from the download queue
	 * @param iRow row number to remove
	 */
	public void removeItem(int iRow) {
		dataTable.removeRow(iRow);
		setSizeDisplay();
		return;
	}
	
	/**
	 * Get the size of the queue
	 * @return number of items in the queue
	 */
	public int size() {
		return dataTable.getRowCount();
	}
	
	/**
	 * Update the size count on the display for how many files are in the queue.
	 */
	public int setSizeDisplay() {
		int iSize = dataTable.getRowCount();
		jlCount.setText("Files in download queue: " + iSize);
		return iSize;
	}
	
	/**
	 * Update how many files have been downloaded.
	 * @param iCount
	 */
	public void updateDownloadCount(int iCount) {
		int iSize = dataTable.getRowCount(), iRow;
		jlCount.setText("Downloaded " + iCount + " files out of " + iSize + ".");
		if (iCount == iSize) {
			lastDLCount = 0;
			hcRender.resetSpecialRows();
			jTable.repaint();
			return;
		}
		for (iRow = lastDLCount; iRow < iCount; iRow++) {
			hcRender.addSpecialRow(iRow);
			jTable.repaint();
		}
		if (iCount != iSize)
			lastDLCount = iCount;
		return;
	}
	
	/**
	 * Check if the queue is empty
	 * @return true if it's empty
	 */
	public boolean isEmpty() {
		boolean isEmpty = true;
		if (dataTable.getRowCount() > 0)
			isEmpty = false;
		return isEmpty;
	}

	/**
	 * Show or hide the queue window.  Center the window over the control panel the first time
	 * it opens.
	 * @param isVisible true to show, false to hide it.
	 */
	public void setVisible(boolean isVisible) {
		this.isVisible = isVisible;
		if (isVisible && firstOpen && parentFrame != null) {
			GUIUtils.centerWindowOnWindow(parentFrame, jFrame);
			firstOpen = false;
		}
		jFrame.setVisible(isVisible);
		return;
	}
	
	/**
	 * Deactivate all the componentents on this window.
	 */
	public void activate() {
		setActive(true);
		return;
	}
	
	/**
	 * Activate all the componentents on this window.
	 */
	public void deactivate() {
		setActive(false);
		return;
	}
	
	/**
	 * Do the actual work for activate() and deactivate().  Turn components on or off.
	 * @param isActive true turns on components, false turns them off
	 */
	private void setActive(boolean isActive) {
		jbClear.setEnabled(isActive);
		jbRemove.setEnabled(isActive);
		jbHelp.setEnabled(isActive);
		jbDownload.setEnabled(isActive);
//		jbClose.setEnabled(isActive);
		return;
	}
	
	/**
	 * Send control to whatever method is going to handle the process.
	 */
	public void actionPerformed(ActionEvent aEvent) {
		int[] iRows = null;
		Object oSource = aEvent.getSource();
		if (oSource == jbClear) {
			reset();
		} else if (oSource == jbRemove) {
			iRows = jTable.getSelectedRows();
			removeItems(iRows);
		} else if (oSource == jbDownload) {
			otrControl.downloadEpisodes(true);
		} else if (oSource == jbClose) {
			setVisible(false);
			jFrame.setVisible(false);
		} else if (oSource == jbHelp) {
			if (firstHelp)
				helpWindow.centerOnWindow(jFrame);
			firstHelp = false;
			helpWindow.setVisible(true);
		}
		return;
	}
	
	/**
	 * Set up the GUI.  Do it down here, as part of constructor to keep all the
	 * messy GUI stuff out of the way.
	 */
	private void guiSetup() {
		int x, y;
		TableColumnModel tcModel;
		TableColumn thisColumn;
		Color bgColor = Color.MAGENTA;
		getJFrame();
		jTable.setColumnSelectionAllowed(false);
		jTable.setRowSelectionAllowed(true);
		jTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		jTable.setShowGrid(true);
		jbClear.addActionListener(this);
		jbRemove.addActionListener(this);
		jbDownload.addActionListener(this);
		jbClose.addActionListener(this);
		jbHelp.addActionListener(this);
		jTable.getTableHeader().setBackground(Color.yellow);
		dataTable = (DefaultTableModel) jTable.getModel();
		tcModel = jTable.getColumnModel();
		for (x = 0; x < colNames.length; x++) {
			dataTable.addColumn(colNames[x]);
			thisColumn = tcModel.getColumn(x);
			thisColumn.setPreferredWidth(100);
			if (hiddenColumns.hasKey(colNames[x])) {
				thisColumn.setPreferredWidth(0);
			}
		}
		for (x = 0; x < hiddenColNames.length; x++) {
			tcModel = jTable.getColumnModel();
			for (y = 0; y < tcModel.getColumnCount(); y++) {
				thisColumn = tcModel.getColumn(y);
				String colName = (String) thisColumn.getHeaderValue();
				if (hiddenColumns.hasKey(colName)) {
					jTable.removeColumn(thisColumn);
				}
			}
		}
		hcRender = new HalCellRenderer();
		jTable.setDefaultRenderer(Object.class, hcRender);
		hcRender.colorByRow(true);
		hcRender.setSpecialBackgroundColor(bgColor);
//		hcRender.addSpecialRow(1);
		return;
	}

//	=============================================================================
//	Beyond here, there be dragons
//	=============================================================================
		
	/**
	 * This method initializes jFrame	
	 * 	
	 * @return javax.swing.JFrame	
	 */
	private JFrame getJFrame() {
		if (jFrame == null) {
			jFrame = new JFrame();
			jFrame.setSize(new Dimension(680, 324));
			jFrame.setTitle("Old Time Radio Show Downloader: Download Queue");
			jFrame.setContentPane(getJContentPane());
		}
		return jFrame;
	}

	/**
	 * This method initializes jContentPane	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getJContentPane() {
		if (jContentPane == null) {
			jLabel = new JLabel();
			jLabel.setText("Download Queue:");
			jContentPane = new JPanel();
			jContentPane.setLayout(new BoxLayout(getJContentPane(), BoxLayout.Y_AXIS));
			jContentPane.add(getJPanel1(), null);
		}
		return jContentPane;
	}

	/**
	 * This method initializes jScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getJScrollPane() {
		if (jScrollPane == null) {
			jScrollPane = new JScrollPane();
			jScrollPane.setViewportView(getJTable());
		}
		return jScrollPane;
	}

	/**
	 * This method initializes jTable	
	 * 	
	 * @return javax.swing.JTable	
	 */
	private JTable getJTable() {
		if (jTable == null) {
			jTable = new JTable();
		}
		return jTable;
	}

	/**
	 * This method initializes jbClear	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getJbClear() {
		if (jbClear == null) {
			jbClear = new JButton();
			jbClear.setText("Clear Queue");
		}
		return jbClear;
	}

	/**
	 * This method initializes jbRemove	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getJbRemove() {
		if (jbRemove == null) {
			jbRemove = new JButton();
			jbRemove.setText("Remove File(s)");
		}
		return jbRemove;
	}

	/**
	 * This method initializes jbHelp	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getJbHelp() {
		if (jbHelp == null) {
			jbHelp = new JButton();
			jbHelp.setText("Help");
		}
		return jbHelp;
	}

	/**
	 * This method initializes jbDownload	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getJbDownload() {
		if (jbDownload == null) {
			jbDownload = new JButton();
			jbDownload.setText("Download Queue");
		}
		return jbDownload;
	}

	/**
	 * This method initializes jButtons	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getJButtons() {
		if (jButtons == null) {
			jButtons = new JPanel();
			jButtons.setLayout(new BoxLayout(getJButtons(), BoxLayout.X_AXIS));
			jButtons.add(getJbClear(), null);
			jButtons.add(getJbRemove(), null);
			jButtons.add(getJbDownload(), null);
			jButtons.add(getJbHelp(), null);
			jButtons.add(getJbClose(), null);
		}
		return jButtons;
	}

	/**
	 * This method initializes jPanel1	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getJPanel1() {
		if (jPanel1 == null) {
			jlCount = new JLabel();
			jlCount.setText("Files in Download Queue:");
			jlCount.setAlignmentX(Component.CENTER_ALIGNMENT);
			jPanel1 = new JPanel();
			jPanel1.setLayout(new BoxLayout(getJPanel1(), BoxLayout.Y_AXIS));
			jPanel1.add(jLabel, null);
			jPanel1.add(getJScrollPane(), null);
			jPanel1.add(jlCount, null);
			jPanel1.add(getJButtons(), null);
		}
		return jPanel1;
	}

	/**
	 * This method initializes jbClose	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getJbClose() {
		if (jbClose == null) {
			jbClose = new JButton();
			jbClose.setText("Close Window");
		}
		return jbClose;
	}

}  //  @jve:decl-index=0:visual-constraint="31,31"
