package com.hal.oldtimeradio;
/**
 * Program: Old Time Radio Show Downloader
 * Copyright (C) 2007 by Hal Vaughan, hal@halblog.com  
 * Licensed under the General Public License (GPL) Version 2.0
 * See http://halblog.com for more information.
 * 
 * The site http://archive.org has, at my guess, over 10,000 episodes of old time radio
 * shows available for downloading.  This program is designed to make it easy to download these
 * shows more easily than using the web site.
 */
import java.io.File;
import java.util.Arrays;
import java.util.LinkedList;

import com.hal.conf.GeneralConfig;
import com.hal.util.HalFile;
import com.hal.util.StringHashMap;

/**
 * Keep track of all the different shows (or series) available to download in archive.org.
 * We only care about OTR (old time radio) shows.  We keep the list of the titles along with
 * what's needed to get the episode listings and where those would be stored locally.
 */
public class ShowData {

	public LinkedList allTitles;
	
	private String otrDirectory, epFilePath, showList;
	
	private StringHashMap myURL, myDir, myEpFile, myDesc, uniqueTitles;
	
	GeneralConfig myConfig;
	
	/**
	 * Basic contstructor.  Nothing to see here.  Move along.
	 * @param programConfig the main config object for this program.
	 */
	public ShowData(GeneralConfig programConfig) {
		myConfig = programConfig;
		reset();
	}
	
	/**
	 * Reset all the data so we have no shows listed and can start fresh.
	 */
	public void reset() {
		allTitles = new LinkedList();
		myURL = new StringHashMap();
		myDir = new StringHashMap();
		myEpFile = new StringHashMap();
		myDesc = new StringHashMap();
		uniqueTitles = new StringHashMap();
		otrDirectory = myConfig.get("OTRDirectory");
		showList = otrDirectory + File.separator + myConfig.get("ShowListFile");
		return;
	}
	
	/**
	 * Automatically load all the shows in the file specified in the main config.
	 */
	public void loadShows() {
		String sData;
		String[] aData;
		reset();
		File fList = new File(showList);
//		System.out.println("Trying to load shows.  File: " + showList);
		if (fList.isFile()) {
			sData = HalFile.fileToString(showList);
			aData = sData.split("\n");
			for (int x = 0; x < aData.length; x++) {
				addShowLine(aData[x]);
			}
		}
		return;
	}
	
	/**
	 * Save a listing of all the shows to disk.
	 */
	public void saveShows() {
		int iPage;
		String[] allTitles = listSortedTitles();
		String sPage = "";
		HalFile fileWriter = new HalFile();
//		System.out.println("Trying to save shows.  File: " + showList);
		fileWriter.openTextFileOut(showList);
		for (iPage = 0; iPage < allTitles.length; iPage++) {
//			sPage = sPage + getTitleLine(allTitles[iPage]);
			sPage = getTitleLine(allTitles[iPage]);
			if (iPage < allTitles.length - 1) {
				sPage = sPage + "\n";
			}
			fileWriter.writeTextOut(sPage);
//			if (iPage == 0) {
//				HalFile.stringToFile(showList, sPage);
//			} else {
//				HalFile.appendStringToFile(showList, sPage);
//			}
		}
		fileWriter.closeTextFileOut();
		return;
	}
	
	/**
	 * Add a new show to the listing of shows.  The other data needed will be calculated.
	 * @param newTitle title of show to add
	 * @param titleURL url for downloading episode lists
	 * @param titleDesc description of show
	 * @return the actual title name used, since we might add (###) to separate some titles.
	 */
	public String newShow(String newTitle, String titleURL, String titleDesc) {
//		int iPoint;
//		iPoint = titleURL.lastIndexOf("/");
//		String directoryName = titleURL.substring(iPoint + 1);
//		String thisTitle = addTitle(newTitle, titleURL, directoryName, titleDesc);
		String thisTitle = addTitle(newTitle, titleURL, titleDesc);
		return thisTitle;
	}
	
	/**
	 * Add a show to our structures by just giving this a line read in from the listing file.  It splits
	 * up the line into fields and calls another routine to add this to our data structures.
	 * @param titleLine tab delimited line of data
	 * @return actual show title that was just added
	 */
	public String addShowLine(String titleLine) {
		String thisTitle = "";
		String[] allParts = titleLine.split("\t");
//		addTitle(allParts[0], allParts[1], allParts[2], allParts[3]);
		String showDesc = "";
		if (allParts.length > 3)
			showDesc = allParts[3];
		addTitle(allParts[0], allParts[1], showDesc);
		return thisTitle;
	}
	
	/**
	 * Add a title to our listing.  If there is a title by this name already existing, then
	 * we increment a number until we find one not already in our list and use that title.  We also
	 * change the original title of the other show that matches to add a (000) on the end of that title
	 * and adjust the other data structures as needed.
	 * @param newTitle  Title of show to add
	 * @param titleURL url of show we're adding
	 * @param directoryName directory name
	 * @param titleDesc description of show
	 * @return finalized title that we used when adding the show.
	 */
//	private String addTitle(String newTitle, String titleURL, String directoryName, String titleDesc) {
	private String addTitle(String newTitle, String titleURL, String titleDesc) {
		int iCount = 0;
		String thisTitle = newTitle, sCount = "000", myDirectory;
		if (uniqueTitles.hasKey(newTitle)) {
			if (allTitles.contains(newTitle)) {
				thisTitle = newTitle + " (" + sCount + ")";
				allTitles.remove(newTitle);
				allTitles.add(thisTitle);
				myURL.put(thisTitle, myURL.get(newTitle));
				myDir.put(thisTitle, myDir.get(newTitle));
				myDesc.put(thisTitle, myDesc.get(newTitle));
				setEpFilePath(thisTitle, myDir.get(newTitle));
			}
			while (true) {
				sCount = "000" + iCount;
				sCount = sCount.substring(sCount.length() - 3);
				thisTitle = newTitle + " (" + sCount + ")";
				if (!allTitles.contains(thisTitle)) {
					break;
				} else {
					iCount++;
				}
			}
		} else {
			uniqueTitles.put(thisTitle, "true");
		}
		allTitles.add(thisTitle);
		myURL.put(thisTitle, titleURL);
		iCount = titleURL.lastIndexOf("/");
		myDirectory = titleURL.substring(iCount + 1);
		myDir.put(thisTitle, myDirectory);
		myDesc.put(thisTitle, titleDesc);
//		setEpFilePath(thisTitle, directoryName);
		setEpFilePath(thisTitle, myDirectory);
//		System.out.println("Adding title: " + thisTitle + ", URL: " + titleURL + ", Directory: " + myDirectory + ", File dir: " + directoryName);
//		System.out.println("Adding title: " + thisTitle + ", URL: " + titleURL + ", Directory: " + myDirectory);
		return thisTitle;
	}
	
	/**
	 * Get the actual episode listing file name and return it.
	 * @param showTitle title of show we're working with
	 * @param directoryName the directory name to be converted to a file name
	 * @return the episode listing file
	 */
	private String setEpFilePath(String showTitle, String directoryName) {
		epFilePath = otrDirectory + File.separator + directoryName + ".csv";
		myEpFile.put(showTitle, epFilePath);
		return epFilePath;
	}
	
	/**
	 * Get the line to write to the file that lists all the titles.  This is one line for only one
	 * title.
	 * @param title show to get a line of data for
	 * @return the line of data, separated by tabs
	 */
	public String getTitleLine(String title) {
		String titleLine = title + "\t" + myURL.get(title) + "\t" + myDir.get(title) + "\t" + myDesc.get(title);
		return titleLine;
	}
	
	/**
	 * Get a list of the titles.  Do not sort.
	 * @return array of all titles.
	 */
	public String[] listTitles() {
		String[] aTitles = new String[allTitles.size()];
		aTitles = (String[]) allTitles.toArray(aTitles);
		return aTitles;
 	}

	/**
	 * Get the titles, but sort them.
	 * @return sorted array of all titles
	 */
	public String[] listSortedTitles() {
		String[] aTitles = new String[allTitles.size()];
		aTitles = (String[]) allTitles.toArray(aTitles);
		Arrays.sort(aTitles);
		return aTitles;
 	}
	
	/**
	 * Get the download URL for the episodes list for a show.
	 * @param showTitle show to check
	 * @return URL for downloading the XML file listing the episodes
	 */
	public String getURL(String showTitle) {
		String sValue = myURL.get(showTitle);
		return sValue;
	}
	
	/**
	 * Get the directory name of where a show's downloads will be stored.  It's also the base
	 * of the file that will have a listing of all the episodes. 
	 * @param showTitle show to check
	 * @return directory name
	 */
	public String getDirectory(String showTitle) {
		String sValue = myDir.get(showTitle);
		return sValue;
	}
	
	/**
	 * Get the episode file name of where a show's episode list is.
	 * @param showTitle show to check
	 * @return file name
	 */
	public String getEpisodeFile(String showTitle) {
		String sValue = myEpFile.get(showTitle);
		return sValue;
	}
	
	/**
	 * Get the description of a show
	 * @param showTitle show to get
	 * @return text description of that show
	 */
	public String getDescription(String showTitle) {
		String sValue = myDesc.get(showTitle);
		return sValue;
	}
	
	/**
	 * Check if the episodes for a show have been downloaded.
	 * @param showTitle name of show to check
	 * @return true if we have an episode listing.
	 */
	public boolean isDownloaded(String showTitle) {
		boolean dlFlag = false;
		String epFile = myEpFile.get(showTitle);
		if (epFile == null) {
			String titleURL = myURL.get(showTitle); 
			int iCount = titleURL.lastIndexOf("/");
			String myDirectory = titleURL.substring(iCount + 1);
			setEpFilePath(showTitle, myDirectory);
		}
		File fEpFile = new File(epFile);
		dlFlag = fEpFile.exists();
		return dlFlag;
	}
}
