package com.hal.oldtimeradio;

import java.io.File;
import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JFrame;

import com.hal.conf.GeneralConfig;
import com.hal.gui.GUIUtils;
import com.hal.gui.HalProgressMonitor;
import com.hal.networking.WebIO;
import com.hal.util.HalFile;
import com.hal.util.HalUtil;

/**
 * Handle all the web operations for downloading old time radio shows.
 * @author hal
 *
 */
public class WebOps {

	boolean abortOp = false, isAutomated = false;
	int downloadDelay = 0, downloadRetry = 5;
	String searchURL, baseURL, dlBaseURL, lastErrorShow = "";
	LinkedList qError;
	JFrame parentFrame;
	
	GeneralConfig myConfig;
	WebIO webIO;
	GUIUtils guiUtils;
	HalProgressMonitor pMonitor;
	HalProgressMonitor fileMonitor;
	
	OTRDownload otrControl;
	ShowData allShows;
	EpisodeData allEps, myEps;
	TagEditor tagEditor;
	
	/**
	 * Just a basic constructor.  Set all variables to initial states.  Nothing to see here.  Move along.
	 * @param programConfig generic General Config config object
	 */
	public WebOps(GeneralConfig programConfig) {
		myConfig = programConfig;
		otrControl = (OTRDownload) myConfig.getObject("otrcontrol");
		parentFrame = otrControl.jFrame;
		fileMonitor = otrControl.fileMonitor;
		tagEditor = otrControl.tagEditor;
		allShows = (ShowData) myConfig.getObject("allshows");
		allEps = (EpisodeData) myConfig.getObject("allepisodes");
		myEps = new EpisodeData(myConfig, allShows);
		searchURL = myConfig.get("SearchForm");
		baseURL = myConfig.get("BaseURL");
		dlBaseURL = myConfig.get("DownloadURL");
		downloadRetry = Integer.parseInt(myConfig.get("DownloadRetry"));
		downloadDelay = Integer.parseInt(myConfig.get("DownloadDelay"));
		webIO = new WebIO();
		webIO.appendData = true;
		guiUtils = new GUIUtils(myConfig);
		resetErrorQueue();
		pMonitor = new HalProgressMonitor(parentFrame, "Downloading show titles");
		guiUtils.tokenMonitor = pMonitor;
	}
	
	/**
	 * Reset the error or warning queue.
	 */
	public void resetErrorQueue() {
		qError = new LinkedList();
		return;
	}
	
	/**
	 * Get a list of all the errors or warnings since the last reset.  Each line is
	 * tab separated.  Fields are show/episode/epfile/error
	 */
 	public String[] getErrorQueue() {
 		String[] allErrors = new String[qError.size()];
 		allErrors = (String[]) qError.toArray(allErrors);
 		return allErrors;
 	}
 	
 	/**
 	 * Add an error to the list.
 	 * @param showName name of radio series
 	 * @param epFile episode name
 	 * @param errorMsg error explanation
 	 */
 	private void addError(String showName, String epFile, String errorMsg) {
 		String sLine;
 		if (!lastErrorShow.equals(showName)) {
 			myEps.setNewShow(showName);
 			myEps.loadEpisodes(showName);
 			lastErrorShow = showName;
 		}
 		sLine = showName + "\t" + myEps.getTitle(epFile) + "\t" + allShows.getDirectory(showName) + "\t" + epFile + "\t" + errorMsg;
 		qError.add(sLine);
 		return;
 	}
	
	/**
	 * Get a list of all the OTR shows (or series) available on archive.org 
	 * Start by telling Archive.org that we're getting a list of OTR show titles, sorted
	 * by title, and specifying page 1.  We scan the page returned, pull out all the show titles,
	 * the links to them, and the descriptions.  We use this info to make a directory name where info
	 * on that show will be stored, then save all the info we have.  When we're done, we sort the
	 * names (in case something went wrong with the sorting previously), then write them to a file.  Each
	 * OTR show gets one line, starting with a "0" to show we're not going to scan that show (which, when
	 * changed to a 1, says to scan that show), the show title, the URL to that page, the directory where
	 * we'll be storing those shows, and the description of the show.
	 */
	public void downloadShowList() {
		otrControl.deactivate();
		pMonitor.activate("Downloading show titles");
		boolean searchFlag = true;
		int iPage = 1, iStart, iEnd;
		String statText, sPage, webPage, sMatch, sNavLine, totalPages = "unknown";		
		String sData = "<td.*?class=\"hitCell\".*?</td>";
		String sTitle = "href.*?</a>";
		String sLink = "href=\".*?\">";
		String sDesc = "<br.*?>.*?<br.*?>";
		String sNav = "<td colspan=\"2\" class=\"pageRow\">.*?</td>";
		String sNext = ">Next<";
		String sNavLink = "<a.*?>.*?</a>";
		String sLastPage = "page=[0-9]+";
		String sSpaces = "^ *$";
		String sNBSpace = "&(nbsp|NBSP);";
		String sTag = "<.*?>";
		Pattern pData = Pattern.compile(sData);
		Pattern pTitle = Pattern.compile(sTitle);
		Pattern pLink = Pattern.compile(sLink);
		Pattern pDesc = Pattern.compile(sDesc);
		Pattern pNav = Pattern.compile(sNav);
		Pattern pNext = Pattern.compile(sNext);
		Pattern pNavLink = Pattern.compile(sNavLink);
		Pattern pLastPage = Pattern.compile(sLastPage);
		Pattern pSpaces = Pattern.compile(sSpaces);
		Pattern pTag = Pattern.compile(sTag);
		Pattern pNBSpace = Pattern.compile(sNBSpace);
		Matcher pageMatch, lineMatch;
		allShows.reset();
		webIO.appendData = true;
		webIO.setURL(searchURL);
		System.out.println("Downloading old time radio show titles.");
		abortOp = false;
		while (searchFlag) {
			searchFlag = false;
			sPage = String.valueOf(iPage);
			webIO.clearFormData();
			webIO.addFormValue("sort", "title");
			webIO.addFormValue("query", "collection%3Aoldtimeradio");
			webIO.addFormValue("page", sPage);
			sTitle = "";
			sLink = "";
			sDesc = "";

			statText = webIO.getFormURL();
			if (webIO.getAbortStatus(statText)) {
				abortOp = true;
				webIO.resolveAction(statText);
				return;
			}
			if (webIO.getErrorStatus(statText)) {
				String sError = webIO.getErrorMessage(statText);
				guiUtils.setParentFrame(parentFrame);
				guiUtils.showInfoDialog("Error downloading show titles.  No data or changes recorded.\nError: " + sError);
				return;
			}
			webPage = webIO.getOutputText(statText);
			webIO.resolveAction(statText);
//			System.out.println("Web page: " + webPage);
			pageMatch = pNav.matcher(webPage);
			if (pageMatch.find()) {
//				System.out.println("Nav link found.");
				sNavLine = pageMatch.group();
//				System.out.println("Navline: " + sNavLine);
				lineMatch = pNext.matcher(sNavLine);
				if (lineMatch.find()) {
					searchFlag = true;
//					System.out.println("Found un-last page verification.");
				}
				if (iPage == 1) {
					pageMatch = pNavLink.matcher(sNavLine);
					while(pageMatch.find()) {
						sMatch = pageMatch.group();
//						System.out.println("Match link: " + sMatch);
						if (sMatch.contains(">Last<")) {
							lineMatch = pLastPage.matcher(sMatch);
							if (lineMatch.find()) {
								sMatch = lineMatch.group();
//								System.out.println("Found last link: " + sMatch);
								iStart = sMatch.indexOf("=");
								if (iStart < sMatch.length()) {
									totalPages = sMatch.substring(iStart + 1);
								}
							}
						}
					}
				}
			}
			pMonitor.setTaskLength(0, Integer.parseInt(totalPages) + 1);
			pageMatch = pData.matcher(webPage);
			while (pageMatch.find()) {
				sMatch = pageMatch.group();
				lineMatch = pLink.matcher(sMatch);
				sLink = ""; sTitle = ""; sDesc = "";
				if (lineMatch.find()) {
					sLink = lineMatch.group();
					iStart = sLink.indexOf("\"");
					sLink = baseURL + sLink.substring(iStart + 1, sLink.length() - 2);
					lineMatch = pTitle.matcher(sMatch);
					if (lineMatch.find()) {
						sTitle = lineMatch.group();
						iStart = sTitle.indexOf(">");
						iEnd = sTitle.lastIndexOf("<");
						sTitle = sTitle.substring(iStart + 1, iEnd);
					}
					lineMatch = pDesc.matcher(sMatch);
					lineMatch.reset();
					if (lineMatch.find()) {
						sDesc = lineMatch.group();
						iStart = sDesc.indexOf(">");
						iEnd = sDesc.lastIndexOf("<");
						sDesc = sDesc.substring(iStart + 1, iEnd);
						lineMatch = pNBSpace.matcher(sDesc);
						lineMatch.reset();
						sDesc = lineMatch.replaceAll(" ");
						lineMatch = pTag.matcher(sDesc);
						if (lineMatch.find()) {
							if (lineMatch.groupCount() == 0) sDesc = "";

						}
						if (sDesc.equals("")) {
							sDesc = "(Description not available.)";
						} else {
							lineMatch = pSpaces.matcher(sDesc);
							if (lineMatch.find()) {
								sDesc = "(Description not available.)";
							}
						}
					}
//					System.out.println("\tLink: " + sLink + ", Title: " + sTitle + ", Desc: " + sDesc);
					allShows.newShow(sTitle, sLink, sDesc);
//					System.out.println("Title: " + sTitle + ", Directory: " + showDir);
				}
			}
//			searchFlag = false;
			System.out.println("Page " + sPage + " of " + totalPages + " pages downloaded.");
//			if (iPage < Integer.parseInt(totalPages)) {

//			}
			pMonitor.updateProgressBar(iPage + 1);
			iPage++;
//			System.out.println("Page number: " + iPage);
			if (pMonitor.isCancelled) {
				otrControl.activate();
				pMonitor.deactivate();
				otrControl.loadShowList();
				abortOp = true;
				break;
			}
		}
		if (abortOp) {
//			System.out.println("Downloading show titles aborted.  No changes made to current data.");
			guiUtils.setParentFrame(parentFrame);
			guiUtils.showInfoDialog("Downloading show titles aborted.\nNo changes made to current data.");
		} else {
			System.out.print("All done with show titles.  Sorting and saving...");
			allShows.saveShows();
			pMonitor.updateProgressBar(iPage);
			otrControl.activate();
			pMonitor.deactivate();
			otrControl.loadShowList();
			System.out.println("Show titles saved.");
		}
		abortOp = false;
		return;
	}
	
	/**
	 * Get the list of episodes for one particular show name.  Get the info
	 * on the show from AllShows, use it to find the URL link to the XML file
	 * that lists all the shows, then download and parse it, store it and write
	 * it out to that show's directory.
	 * @param showName name of the show to download episodes from
	 */
	public void downloadEpisodeList(String showName) {
		int iPoint = 0;
		String thisDir, xmlURL, statText, sData;
		thisDir = allShows.getDirectory(showName);
		xmlURL = dlBaseURL + "/" + thisDir + "/" + thisDir + "_files.xml";
//		String sData = webIO.getURL(xmlURL);
//		statText =  webIO.getURL(xmlURL);
		statText = webIO.getURL(xmlURL, false);
		if (webIO.getAbortStatus(statText)) {
			guiUtils.setParentFrame(parentFrame);
			guiUtils.showInfoDialog("Downloading episode titles for radio show aborted.\nShow title: " + showName);
			webIO.resolveAction(statText);
			return;
		}
		if (webIO.getErrorStatus(statText)) {
			String sError = webIO.getErrorMessage(statText);
			guiUtils.setParentFrame(parentFrame);
			guiUtils.showInfoDialog("Error downloading episode titles for show: " + showName + "\nError: " + sError);
			webIO.resolveAction(statText);
			return;
		}
		sData = webIO.getOutputText(statText);
		webIO.resolveAction(statText);
//		String sData = HalFile.fileToString("/home/hal/OldTimeRadio/OTRR_Dimension_X_Singles_files.xml");
//		System.out.println("XML downloaded, Length: " + sData.length());
//		System.out.println("Data:\n" + sData);
		String sFile, sName, sSource, sFormat, sLicense;
		String[] aEps;
		String rexFile = "<file name.*?source.*?</file>";
		String rexName = "<file.*?name=\".*?\"";
		String rexSource = "<file.*?source=\".*?\"";
		String rexFormat = "<format>.*?</format>";
		String rexLicense = "<license>.*?</license>";
		Pattern pFile = Pattern.compile(rexFile, Pattern.MULTILINE | Pattern.DOTALL);
		Pattern pName = Pattern.compile(rexName);
		Pattern pSource = Pattern.compile(rexSource);
		Pattern pFormat = Pattern.compile(rexFormat);
		Pattern pLicense = Pattern.compile(rexLicense);
		Matcher fileMatch = pFile.matcher(sData);
		Matcher itemMatch;
		allEps.reset();
		while (fileMatch.find()) {
//			System.out.println("Found a match!");
			sName = ""; sSource = ""; sFormat = ""; sLicense = "";
			sFile = fileMatch.group();
			itemMatch = pSource.matcher(sFile);
			if (itemMatch.find()) {
				sSource = itemMatch.group();
				iPoint = sSource.indexOf("source=\"", iPoint);
				iPoint = sSource.indexOf("\"", iPoint);
				sSource = sSource.substring(iPoint + 1, sSource.length() - 1);
			}
			if (sSource.equalsIgnoreCase("metadata")) {
				continue;
			}
			itemMatch = pName.matcher(sFile);
			if (itemMatch.find()) {
				sName = itemMatch.group();
				iPoint = sName.indexOf("\"");
				sName = sName.substring(iPoint + 1, sName.length() - 1);
//				System.out.println("Type: " + sSource + ", File: " + sName);
			}
			itemMatch = pFormat.matcher(sFile);
			if (itemMatch.find()) {
				sFormat = itemMatch.group();
				sFormat = sFormat.substring(sFormat.indexOf(">") + 1, sFormat.lastIndexOf("<"));
			}
			itemMatch = pLicense.matcher(sFile);
			if (itemMatch.find()) {
				sLicense = itemMatch.group();
				sLicense = sLicense.substring(sLicense.indexOf(">") + 1, sLicense.lastIndexOf("<"));				
			}
			if (sFormat.toLowerCase().contains("m3u") || sFormat.toLowerCase().contains("metadata") || sName.endsWith("xml"))
				continue;
			allEps.newEpisode(sName, showName, sFormat, sLicense);
//			System.out.println("Name: " + sName + ", Show Title: " + showName + ", Source: " + sSource + ", Format: " + sFormat + ", License: " + sLicense);
//			System.out.println("\tFile URL: " + sURL);
		}
		sData = "";
		aEps = allEps.listSortedFiles();
		for (iPoint = 0; iPoint < aEps.length; iPoint++) {
			sData = sData + allEps.getEpisodeLine(aEps[iPoint]);
			if (iPoint < aEps.length - 1) {
				sData = sData + "\n";
			}
		}
		sFile = allShows.getEpisodeFile(showName);
		HalFile.stringToFile(sFile, sData);
		return;
	}
	
	/**
	 * Do the actual download of an episode file (usually MP3 files).  If all goes well,
	 * download the file and tag it.  If it doesn't download, try again until we reach the
	 * limit (which is downloadRetry).  The downloadDelay variable tells us how long to wait
	 * (in minutes) between each retry.  If the download is complete and we can't tag the file,
	 * we still save it, just untagged, and we warn the user about it.
	 * NOTE: We get more parms than we need for speed.  Even though loading the ep data doesn't take
	 * much time, it takes time, so by getting an extra 2 parms, that can speed up downloading.
	 * @param showName the name of the show the episode is from
	 * @param epFile the file name (no path) of the episode
	 * @param localFile the local file (including path) of the episode
	 * @param webURL the url where the episode is on the web
	 */
	public void downloadAndTagFile(String showName, String epFile, String localFile, String webURL) {
		boolean isDone = false, isDownloaded = false, tagDone = false, isAborted = false, isError = false;
		int dlCount = 0;
		String statText = "", localDir, sLine;
		localDir = localFile.substring(0, localFile.lastIndexOf(File.separator));
		isAutomated = otrControl.isAutomated;
		File fDir = new File(localDir);
		if (fileMonitor == null) {
			fileMonitor = otrControl.fileMonitor;
		}
		if (!fDir.exists()) {
			fDir.mkdirs();
		}
//		if (!lastShow.equals(showName)) {
//			myEps.setNewShow(showName);
//			myEps.loadEpisodes(showName);
//			lastShow = showName;
//		}
		while (!isDone) {
			dlCount = 0;
			while (!isDownloaded) {
				fileMonitor.setData("Downloading from radio show: " + showName, "<html><center>Downloading episode file:<br>" + epFile + "</center></html>");
				statText = webIO.getBinaryURL(webURL, false, fileMonitor);
				fileMonitor.reset();
				isAborted = webIO.getAbortStatus(statText);
				if (isAborted) {
					guiUtils.setParentFrame(parentFrame);
					guiUtils.showInfoDialog("Downloading file aborted.\nNo data saved or files changed.");
					webIO.resolveAction(statText);
					addError(showName, epFile, "File download aborted.  No data or changes saved or made.");
					return;
				}
				isError = webIO.getErrorStatus(statText);
				if (isError) {
					dlCount++;
					if (dlCount > downloadRetry && downloadRetry != 0) {
//Some dialog if we're not in an unattended mode
						checkAndWarn("One file could not be downloaded.\nFile name: " + localFile, false);
						addError(showName, epFile, "Download unsuccessful.  Error given: " + webIO.getErrorMessage(statText));
						webIO.resolveAction(statText);
						return;
					}
					if (downloadDelay > 0) {
						HalUtil.sleep(downloadDelay * 1000);
					}
					continue;
				}
				sLine = HalFile.move(statText, localFile);
				if (!sLine.equals("")) {
					dlCount++;
					checkAndWarn("Could not create file for download.\nFile name: " + localFile + "\nError: " + sLine, false);
					addError(showName, epFile, "Could not create local file.  Error: " + sLine);
					webIO.resolveAction(statText);
					HalFile.delete(localFile);
					if (dlCount <= downloadRetry && checkAndWarn("Make another attempt?", true)) {
						if (downloadDelay > 0 && isAutomated) {
							HalUtil.sleep(downloadDelay * 1000);
						}
						continue;
					}
					return;
				}
//				System.out.println("Finished download and no glitches showed up!");
				isDownloaded = true;
			}
			webIO.resolveAction(statText);
			tagDone = false;
			while (!tagDone) {
				tagDone = tagEditor.setFileTags(showName, epFile, localFile, webURL);
				if (!tagDone) {
//					System.out.println("Not Tagged");
					boolean bCheck = checkAndWarn("Tags could not be added to a file\nFile name: " + localFile + "\nRe-download and try again?" , true);
					if (isAutomated || !bCheck) {
						addError(showName, epFile, "Could not add tags to file.");
						tagDone = true;
					} else {
						isDownloaded = false;
					}
					break;
				}
//				System.out.println("Tagged and done!");
			}
			if (tagDone && isDownloaded)
				isDone = true;
		}
		System.out.println("Finished downloading file: " + epFile);
		return;
	}	
	
	/**
	 * Check to see if we warn the user or just continue.  In many cases this program
	 * may be run in an unattended mode.  If that's the case, then there are many times
	 * we don't ask for input, but just continue.
	 * @param sMsg
	 */
	public boolean checkAndWarn(String sMsg, boolean bYesNo) {
		boolean bContinue = true;
		if (isAutomated)
			return true;
		if (bYesNo) {
			guiUtils.setParentFrame(parentFrame);
			bContinue = guiUtils.showYesNoDialog(sMsg);
		} else {
			guiUtils.showInfoDialog(sMsg);
		}
		
		return bContinue;
	}

}
