package com.hal.util;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;


/**
 * Like a HashMap, but with only String as a key and int as a value.  The idea is to make it easy to keep track
 * of a lot of different counters.  Each counter and be incremented, added to, rest, or set to a specific number.
 * All counters can be reset at once, incremented at once, or added to with the same number at once.
 * @author hal
 *
 */
public class IndexHashMap implements Serializable {
	
	final static long serialVersionUID = 1L;
	
	/**
	 * Initialized at 0 by default, can be changed.  Any new counter is created with this value and
	 * when any counter is reset, it is reset to this value.
	 */
	public int resetValue = 0;

	HashMap myHash;
	
	/**
	 * Create a StringHashMap.
	 */
	public IndexHashMap() {
		myHash = new HashMap();
		resetAll();
	}
	
	/**
	 * Clear all values in this StringHashMap
	 *
	 */
	public void resetAll() {
		int x;
		String[] allKeys = rawKeys();
		for (x = 0; x < allKeys.length; x++) {
			reset(allKeys[x]);
		}
		return;
	}
	
	/**
	 * Clear the value for one counter (as in reset it to 0).
	 * @param sName name of counter to reset
	 */
	public void reset(String sName) {
		create(sName);
		myHash.put(sName, Integer.valueOf(resetValue));
		return;
	}
	
	/**
	 * Increment all the counters by one.
	 */
	public void incAll() {
		int x, iVal;
		String[] allKeys = rawKeys();
		for (x = 0; x < allKeys.length; x++) {
			iVal = ((Integer) myHash.get(allKeys[x])).intValue();
			iVal++;
			myHash.put(allKeys[x], Integer.valueOf(iVal));
		}
		return;
	}
	
	/**
	 * Increment the named counter by one.
	 * @param sName counter ot increment
	 */
	public void inc(String sName) {
		int iVal;
		create(sName);
		iVal = ((Integer) myHash.get(sName)).intValue();
		iVal++;
		myHash.put(sName, Integer.valueOf(iVal));
		return;
	}
	
	/**
	 * Add the specified number to all counters
	 * @param iAdd number to add
	 */
	public void addAll(int iAdd) {
		int x, iVal;
		String[] allKeys = rawKeys();
		for (x = 0; x < allKeys.length; x++) {
			iVal = ((Integer) myHash.get(allKeys[x])).intValue();
			iVal = iVal + iAdd;
			myHash.put(allKeys[x], Integer.valueOf(iVal));
		}
		return;
	}
	
	/**
	 * Add the specified number to a particular counter
	 * @param sName counter to add to
	 * @param iAdd amount to add
	 */
	public void add(String sName, int iAdd) {
		int iVal;
		create(sName);
		iVal = ((Integer) myHash.get(sName)).intValue();
		iVal = iVal + iAdd;
		myHash.put(sName, Integer.valueOf(iVal));
		return;
	}
	
	/**
	 * Set all the counters to a specific number.
	 * @param iSet number to set counters to
	 */
	public void setAll(int iSet) {
		int x;
		String[] allKeys = rawKeys();
		for (x = 0; x < allKeys.length; x++) {
			myHash.put(allKeys[x], Integer.valueOf(iSet));
		}
		return;
	}
	
	/**
	 * Set a specified counter to a specific counter.
	 * @param sName counter to set
	 * @param iSet value to set the counter to
	 */
	public void set(String sName, int iSet) {
		create(sName);
		myHash.put(sName, Integer.valueOf(iSet));
		return;
	}
	
	/**
	 * Create an index counter if one by the same name doesn't already exist.  If
	 * there is one already, don't touch it.
	 * @param sName key name
	 * @param iVal value to store
	 */
	public void create(String sName, int iVal) {
		create(sName);
		set(sName, iVal);
		return;
	}
	
	/**
	 * Put an index counter in.
	 * @param sName key name
	 */
	public void create(String sName) {
		if (!myHash.containsKey(sName))
			myHash.put(sName, Integer.valueOf(resetValue));		
		return;
	}
	
	/**
	 * Get a value by specifying the key name
	 * @param sName the key for the value we want
	 * @return the value specified by the key
	 */
	public int get(String sName) {
		int iVal = resetValue;
		if (myHash.containsKey(sName))
			iVal = ((Integer) myHash.get(sName)).intValue();
		return iVal;
	}
	
	/**
	 * Increment a counter, then get the value (like ++int).
	 * @param sName name of counter
	 * @return value after counter is incremented by one
	 */
	public int incGet(String sName) {
		int iVal;
		create(sName);
		iVal = ((Integer) myHash.get(sName)).intValue();
		iVal++;
		myHash.put(sName, Integer.valueOf(iVal));
		return iVal;
	}
	
	/**
	 * Get a counter, then, after that, increment it by one (like int++).
	 * @param sName name of counter
	 * @return value before counter is incremented by one
	 */
	public int getInc(String sName) {
		int iPre, iPost;
		create(sName);
		iPre = ((Integer) myHash.get(sName)).intValue();
		iPost = iPre + 1;
		myHash.put(sName, Integer.valueOf(iPost));
		return iPre;
	}
	
	/**
	 * Remove a value from the StringHashMap
	 * @param sName The key for the value we're removing
	 */
	public void remove(String sName) {
		if (myHash.containsKey(sName))
			myHash.remove(sName);
		return;
	}
	
	/**
	 * Get a sorted list of all the keys. 
	 * @return sorted list of keys
	 */
	public String[] keySet() {
		String[] aKeys = rawKeys();
		Arrays.sort(aKeys);
		return aKeys;
	}
	
	/**
	 * Get the key set without sorting.
	 * @return full key set
	 */
	private String[] rawKeys() {
		int i = 0;
		String[] aKeys = new String[0];
		Set allKeys = myHash.keySet();
		Iterator iPoint = allKeys.iterator();
		aKeys = new String[allKeys.size()];
		while (iPoint.hasNext()) {
			Object nextItem = iPoint.next();
			if (nextItem != null) {
				aKeys[i++] = (String) nextItem;
			}
		}
		return aKeys;
	}
	
	/**
	 * Get the number of keys with values.
	 * @return size of object.
	 */
	public int getLength() {
		int i = myHash.size();
		return i;
	}
	
	/**
	 * Get the number of the keys with values.
	 * @return size of object
	 */
	public int size() {
		return getLength();
	}
	
	/**
	 * Return true if we have a key matching the specified name
	 * @param keyName the key we're checking to see
	 * @return true if the key exists, false if not.
	 */
	public boolean hasKey(String keyName) {
		if (myHash.containsKey(keyName)) return true;
		return false;		
	}
}
